//
//  ViewController.swift
//  KDTree
//
//  Created by Arielle Moro on 07.12.16.
//  Copyright © 2016 Arielle. All rights reserved.
//

import UIKit
import Foundation

class ViewController: UIViewController {
    
    struct Node {
        let xy: [Int]
        var childrenNodes: [Node] //0 = left child node AND 1 = right child node
        func simpleDescription() -> String {
            return "x:\(xy[0]) and y:\(xy[1])"
        }
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        // Do any additional setup after loading the view, typically from a nib.
    }

    override func didReceiveMemoryWarning() {
        super.didReceiveMemoryWarning()
        // Dispose of any resources that can be recreated.
    }
    
    @IBAction func testKDTreeAlgorithm(_ sender: Any) {
        //List of points to put in the k-d tree
        let points: [[Int]] = [[2,3], [5,4], [9,6], [4,7], [8,1], [7,2]]
        let depth = 0
        let rootNode = buildKDTree(currentPoints: points, currentDepth: depth)
        displayKDTree(node: rootNode, direction: "rootNode", depth: depth)
        let searchPoint = [4,8]
        let nearestNode = nearestNeighbourSearch(searchPoint: searchPoint, node: rootNode)
        let searchPointDesc = "x:\(searchPoint[0]) and y:\(searchPoint[1])"
        let nearestNodeDesc = nearestNode.simpleDescription()
        print("Nearest node on \(searchPointDesc) is \(nearestNodeDesc)")
    }
    
    func buildKDTree(currentPoints: [[Int]], currentDepth: Int) -> Node {
        if(currentPoints.count == 0){
            return Node(xy: [], childrenNodes: []) //empty node creation
        }
        let dim = currentPoints[0].count
        let axis = currentDepth % dim
        let newCurrentPoints = currentPoints.sorted{ $0[axis] < $1[axis] }
        let medianIndex = newCurrentPoints.count / 2
        let medianPoint = newCurrentPoints[medianIndex]
        let leftPoints = newCurrentPoints.filter { $0[axis] < medianPoint[axis]}
        let rightPoints = newCurrentPoints.filter { $0[axis] > medianPoint[axis]}
        let leftNode = buildKDTree(currentPoints: leftPoints, currentDepth: currentDepth+1)
        let rightNode = buildKDTree(currentPoints: rightPoints, currentDepth: currentDepth+1)
        return Node(xy: medianPoint,childrenNodes:[leftNode, rightNode])
    }
    
    func displayKDTree(node: Node, direction: String, depth: Int) {
        let nodeDesc = node.simpleDescription()
        print("=> (\(depth) - \(direction)) \(nodeDesc)")
        let leftChildNode = node.childrenNodes[0]
        if leftChildNode.xy.count != 0 {
            displayKDTree(node: leftChildNode, direction: "left", depth: depth+1)
        }
        let rightRightNode = node.childrenNodes[1]
        if rightRightNode.xy.count != 0 {
            displayKDTree(node: rightRightNode, direction: "right", depth: depth+1)
        }
    }
    
    func nearestNeighbourSearch(searchPoint:[Int], node: Node) -> Node {
        let leftChildNode = node.childrenNodes[0]
        let rightChildNode = node.childrenNodes[1]
        if leftChildNode.xy.count == 0 && rightChildNode.xy.count == 0 {
            return node
        }
        //Local exploration
        var scores: [Double] = [0.0,0.0,0.0]
        scores[0] = compareTwoPoints(searchPoint: searchPoint, point: node.xy)
        scores[1] = compareTwoPoints(searchPoint: searchPoint, point: leftChildNode.xy)
        scores[2] = compareTwoPoints(searchPoint: searchPoint, point: rightChildNode.xy)
        //Find the best score
        var bestScore = scores[0]
        var bestNodeIndex = 0
        var index = 1
        while index < 3 {
            if scores[index] < bestScore{
                bestScore = scores[index]
                bestNodeIndex = index
            }
            index = index + 1
        }
        //Find the nearest neighbour
        if(bestNodeIndex == 1){
            //LeftNode
            return nearestNeighbourSearch(searchPoint:searchPoint, node:leftChildNode)
        }else if(bestNodeIndex == 2){
            //RightNode
            return nearestNeighbourSearch(searchPoint:searchPoint, node:rightChildNode)
        }else{
            //bestNodeIndex = 0!
            return node
        }
    }
    
    func compareTwoPoints(searchPoint:[Int], point:[Int]) -> Double {
        return sqrt(pow((Double(searchPoint[0]) - Double(point[0])), 2) + pow((Double(searchPoint[1]) - Double(point[1])), 2))
    }
    
}

